import getpass
import json
import requests

'''
This example scripts allows the user to login to PowerForge, get
a complete list of available projects, and updates the cooling cost
depending on the user needs, before sending it back to PowerForge.

This example is tailored for designs with "Simple Rth model" cooling variants,
but can be upgraded to allow for update of all cooling variants.
'''

# Authentication method, return the auth token
def auth(email, password):
    endpoint = 'https://powerforge.powerdesign.tech/powerforge/auth/obtain/'
    payload = { "email": email, "password": password }
    resp = requests.post(url=endpoint, json=payload)
    if not resp.ok:
        # This means something went wrong.
        raise Exception('GET /auth/obtain/ {}'.format(resp.status_code))
    return resp.json()['token']

# Gets data from PowerForge
def get(auth_token, endpoint):
    headers = {'Authorization': 'JWT ' + auth_token}
    resp = requests.get(url=endpoint, headers=headers)
    if not resp.ok:
        # This means something went wrong.
        raise Exception('GET ' + endpoint + ' : {}'.format(resp.status_code))
    return resp.json()

# Sends data to PowerForge
def put(auth_token, endpoint, payload):
    headers = {'Authorization': 'JWT ' + auth_token}
    resp = requests.put(url=endpoint + "/", headers=headers, json=payload)
    if not resp.ok:
        # This means something went wrong.
        raise Exception('PUT ' + endpoint + ' : {}'.format(resp.status_code))

### Main Script

# Getting authentication token that will be used for all following requests
email = input("User email: ") 
password = getpass.getpass(prompt='Password: ', stream=None) 

print("Connecting...")
auth_token = auth(email, password)

# Getting all projects and displaying their names
print("Getting list of projects...")
projects = get(auth_token, 'https://powerforge.powerdesign.tech/powerforge/api/projects/')
for i, project in enumerate(projects):
    print(i, " : ", project["name"])

# Selecting and getting a project data
index = int(input("Enter number of project to update: "))
project_url = projects[index]["url"]

print("Getting and updating all designs...")
project = get(auth_token, project_url)

# Getting all designs from all sweeps of this project
for i, sweep_url in enumerate(project["sweeps"]):
    sweep_data = get(auth_token, sweep_url)
    for j, design in enumerate(sweep_data["designs"]):
        design_url = design["url"] + 'configuration/raw/'
        raw_design_data = get(auth_token, design_url)

        # Updating design data. This takes the downloaded
        # data and updates it before sending it back to PowerForge.
        #
        # To know all field names for your design, you can display
        # the complete structure of the downloaded design.
        #
        # As an example, the following code takes the first
        # downloaded design, and display its structure and values.
        '''
        if i == 0 and j == 0 :
            print("Example structure from a design of this project:")
            print(json.dumps(raw_design_data, indent=4, sort_keys=True))
        '''

        # Computing new fields values and updating them
        try:
            print("Updating " + design["name"] + "...")
            raw_design_data["cost"] = raw_design_data["cost"] - raw_design_data["heatsinkAutoDesign"]["cost"] + 200
            raw_design_data["heatsinkAutoDesign"]["cost"] = 200

            # Sending data back to PowerForge
            update_url = design["url"] + 'configuration/raw'
            put(auth_token, update_url, raw_design_data)
        except:
            print("> Could not update. Field could not be found or data could not be sent back to PowerForge.")
            pass        

print("Update complete.")