from pathlib import Path, PureWindowsPath

import getpass
import json
import requests
import os

'''
To allow for launching a valid sweep, a sweep specification file is used.
Such file can be downloaded by starting up a sweep specification in the PowerForge software,
setting up all your subsystems variants and clicking the Download button in the upper
bar. This will produce a "specification.json" file that can be modified depending on your needs.

Just place the json file in the same folder as this script before executing it, and the sweep
will be sent and launched into PowerForge.
'''

# Authentication method, return the auth token
def auth(email, password):
    endpoint = 'https://powerforge.powerdesign.tech/powerforge/auth/obtain/'
    payload = { "email": email, "password": password }
    resp = requests.post(url=endpoint, json=payload)
    if not resp.ok:
        # This means something went wrong.
        raise Exception('GET /auth/obtain/ {}'.format(resp.status_code))
    return resp.json()['token']

# Gets data from PowerForge
def get(auth_token, endpoint):
    headers = {'Authorization': 'JWT ' + auth_token}
    resp = requests.get(url=endpoint, headers=headers)
    if not resp.ok:
        # This means something went wrong.
        raise Exception('GET ' + endpoint + ' : {}'.format(resp.status_code))
    return resp.json()

# Sends data to PowerForge
def post(auth_token, endpoint, payload):
    headers = {'Authorization': 'JWT ' + auth_token}
    endpoint = endpoint + "/"
    resp = requests.post(url=endpoint, headers=headers, json=payload)
    if not resp.ok:
        # This means something went wrong.
        raise Exception('POST ' + endpoint + ' : {}'.format(resp.status_code))

### Main Script

# Opening specification file and getting its contents
spec_file = open(str(Path(__file__).absolute().parent / 'specification.json').__str__(), "r")
specification = spec_file.read() 
json_spec = json.loads(specification)

# Getting authentication token that will be used for all following requests
email = input("User email: ") 
password = getpass.getpass(prompt='Password: ', stream=None) 

print("Connecting...")
auth_token = auth(email, password)

# Getting all projects and displaying their names
print("Getting list of projects...")
projects = get(auth_token, 'https://powerforge.powerdesign.tech/powerforge/api/projects')
for i, project in enumerate(projects):
    print(i, " : ", project["name"])

# Selecting and getting a project url
index = int(input("Enter number of project to launch sweep into: "))
project_url = projects[index]["url"]

# Setting sweep specification, name and project to be launch into
endpoint = "https://powerforge.powerdesign.tech/powerforge/api/sweeps"
post_payload = {
    "project": project_url,
    "name": "Name of my sweep",
    "json":  json_spec
}

print("Launching sweep...")
post(auth_token, endpoint, post_payload)

print("Sweep launched. Sweep progress can be checked in PowerForge.")
